define([
    'underscore',
    'backbone',
    'Layout',
    'modules/new-appointment-request/views/direct/clinics/section-view',
    'modules/new-appointment-request/resources/direct/preferred-date/clinics-time-slots-collections',
    'modules/new-appointment-request/resources/direct/custom-messages/collection',
], function(_, Backbone, Layout, ClinicsSectionView, ClinicsTimeSlotsCollections, CustomMessagesCollection) {
    'use strict';

    var view;
    var model;
    var clinicsTimeSlotsCollections;
    var customMessages;
    var layout;


    describe('Clinics Section', function() {

        beforeEach(function() {
            customMessages = new CustomMessagesCollection([
                {
                    'id': 'noClinicRadioButtons',
                    'messageId': 'noClinicRadioButtons',
                    'messageText': 'This is a test.',
                },
            ]);
            layout = new Layout();
            layout.render();
        });


        describe('when there is one valid clinic', function() {
            beforeEach(function() {
                model = new Backbone.Model({typeOfCare: '323'});

                clinicsTimeSlotsCollections = new ClinicsTimeSlotsCollections([
                    {
                        clinicId: '428',
                        clinicName: 'PRIMARY CARE SITE 428',
                        appointmentTimeSlot: [
                            {
                                startDateTime: '05/12/2016 08:00:00',
                                endDateTime: '05/12/2016 08:30:00',
                                bookingStatus: '4',
                                remainingAllowedOverBookings: '0',
                                availability: true,
                            },
                        ],
                    },
                    {
                        clinicId: '425',
                        clinicName: 'PRIMARY CARE SITE 425',
                        appointmentTimeSlot: [],
                    },
                ], {parse: true});

                spyOn(ClinicsSectionView.prototype, 'setClinicQuestionModel').and.callThrough();

                view = new ClinicsSectionView({
                    model: model,
                    clinicsTimeSlotsCollections: clinicsTimeSlotsCollections,
                    clinicsCollection: new Backbone.Collection(),
                    clinicCustomMessage: customMessages.get('noClinicRadioButtons'),
                });

                layout.showChildView('content', view);
            });

            it('calls setClinicQuestionModel on initialize', function() {
                expect(view.setClinicQuestionModel).toHaveBeenCalled();
            });
        });

        describe('getTimeSlots', function() {
            beforeEach(function() {
                view = new ClinicsSectionView({
                    model: new Backbone.Model(),
                    clinicsTimeSlotsCollections: new Backbone.Collection(),
                    clinicsCollection: new Backbone.Collection(),
                    clinicCustomMessage: new Backbone.Model(),
                });
                layout.showChildView('content', view);
            });

            it('returns null when there are no slots', function() {
                expect(view.getTimeSlots(null)).toBeNull();
            });

            it('returns the time slot when it exists', function() {
                var ID = '123';
                var m = new Backbone.Model({id: ID, data: '123'});

                view.slots = new Backbone.Collection([m]);

                expect(view.getTimeSlots(ID).get('data')).toBe('123');
            });
        });

        describe('fetchSlotsSuccess', function() {
            beforeEach(function() {
                view = new ClinicsSectionView({
                    model: new Backbone.Model(),
                    clinicsTimeSlotsCollections: new Backbone.Collection(),
                    clinicsCollection: new Backbone.Collection(),
                    clinicCustomMessage: new Backbone.Model(),
                });
                spyOn(view, 'showErrorMessage').and.callFake(_.noop);
                layout.showChildView('content', view);

            });

            it('shows an error when slots are missing', function() {
                view.fetchSlotsSuccess(new Backbone.Collection());
                expect(view.showErrorMessage).toHaveBeenCalled();
            });

            it('sets a clinic when successful', function() {
                var m = new Backbone.Model({id: '123', data: 'abc', appointmentTimeSlot: ['1']});
                var c = new Backbone.Collection([m]);
                view.clinicList = new Backbone.Collection([m]);
                view.fetchSlotsSuccess(c);

                expect(view.showErrorMessage).not.toHaveBeenCalled();
                expect(view.model.get('clinic').toJSON()).toEqual(m.toJSON());
            });
        });
    });
});
